/*! \file 
===============================================================================
   FILE:         TTApi_FM.h
-------------------------------------------------------------------------------
   DESCRIPTION:  ActiveRadio API
                 Hardware control functions & structures
   AUTHOR:       F.M.Birth, TerraTec Electronic GmbH
-------------------------------------------------------------------------------
   HISTORY:      16-Feb-01  - Reorganization
                              new:     radioGetNumDevs(),radioGetDevCaps(),radioSelect()
                              removed :WSS mixer routines -> this DLL only handles hardware volume/mute control
                 08-Apr-02  - Added support for Cinergy 600 TV
                 09-Oct-04  - Removed outdated error codes
                 18-Oct-04  - Reorganization: 
                              modified error codes
                              separated  RDS and FM tuner API
                              device index required (replaced radioSelect())

   NOTES:        - radioInitialize()/radioUninitalize must be called
                 - Most function return an TTRADIO_E_xxx error code or 
                   TTRADIO_S_OK if successful

   TODO:         - TUNERSTATE: separate usage of read and write values

-------------------------------------------------------------------------------
   COPYRIGHT:    (c) 1999-2004, TerraTec Electronic GmbH. All Rights Reserved.
                 (c) 2004, Friedhelm Birth
=============================================================================*/

#ifndef __IOEXPORT_H__
#define __IOEXPORT_H__

#ifdef DRIVER
    #define RDSLIBAPI
#else // -> not driver
    #include <StdIo.h>   // for FILE type definition
    #include <Windows.h>


    #ifdef DLL_EXPORTS // Logic for export/import
	    #define RDSLIBAPI __declspec(dllexport)            // for function declarations
	    #define RDSLIBDEF extern "C" __declspec(dllexport) // for function definitions
    #else
	    #define RDSLIBAPI __declspec(dllimport)
	    #pragma comment (lib, "Radio_io.lib")
    #endif
#endif // not driver

/*===============================================================
    CONSTANTS    
===============================================================*/   
#define ONE_KHZ   (1000)
#define ONE_MHZ   (1000*ONE_KHZ)

#define FREQ_MIN  ( 87500 * ONE_KHZ) // FM band - lower margin
#define FREQ_MAX  (108000 * ONE_KHZ) // FM band - upper margin

#define FM_IF     ( 10700 * ONE_KHZ) // FM intermediate frequency



/*===============================================================
    ActiveRadio error codes
===============================================================*/   

#define TTSUCCEEDED(Status) ((TTRESULT)(Status) >= 0)
#define TTFAILED(Status)    ((TTRESULT)(Status) <  0)

#define _TTRESULT_TYPEDEF_(_sc) ((TTRESULT)_sc)

typedef enum tagTTRESULT
{

// Basic return values
//-----------------------------------------------------------------------------
   TTRADIO_S_OK                  = (0x00000000L)
,  TTRADIO_S_FALSE               = (0x00000001L)  //!< e.g. nothing to do
,  TTRADIO_E_FAIL                = (0xFFFFFFFFL)

// General errors
//-----------------------------------------------------------------------------
,  TTRADIO_E_UNSUPPORTED         = (0x80000001L) //!< hardware doesn't support this feature
,  TTRADIO_E_INVALIDARG          = (0x80000002L)    
,  TTRADIO_E_NOTIMPL             = (0x80000003L) //!< feature is not implemented (for this hardware)
,  TTRADIO_E_NOINTERFACE         = (0x80000004L) //!< required (COM) interface could not be openend
,  TTRADIO_E_POINTER             = (0x80000005L) //!< NULL pointer
,  TTRADIO_E_HANDLE              = (0x80000006L) //!< NULL or invalid handle

,  TTRADIO_E_NOHARDWARE          = (0x80000007L) //!< hardware or hardware instaance not available
,  TTRADIO_E_SECOND_CALL         = (0x80000008L) //!< calling a function the 2nd time is not allowed
,  TTRADIO_E_INVALID_STATE       = (0x80000009L) //!< function is not available in this state
,  TTRADIO_E_UNINITIALIZED       = (0x8000000AL) //!< called before initialization



// I2C errors
//-----------------------------------------------------------------------------
,  TTRADIO_E_IIC_READ	          = (0x80000010L)
,  TTRADIO_E_IIC_WRITE	          = (0x80000020L)

// Tuning/search errors
//-----------------------------------------------------------------------------

,  TTRADIO_E_INVALID_FREQUENCY   = (0x80000100L) //!< frequency read from the hardware is invalid
,  TTRADIO_E_TIMEOUT             = (0x80000200L) //!< tuner search couldn't find a station in time
,  TTRADIO_E_TOO_CLOSE           = (0x80000300L) //!< found frequency is too close to start position
,  TTRADIO_E_NO_STATION_FOUND    = (0x80000400L)
,  TTRADIO_E_TUNER_BUSY          = (0x80000500L)

// RDS errors
//-----------------------------------------------------------------------------
,  RDS_INCORRECTABLE_BLOCK       = (0x80001000L)
,  RDS_UNKNOWN_RDS_DECODER       = (0x80002000L) //!< cannot call a SAA6588 specify function on this decoder

}TTRESULT;

/*=============================================================================
    TYPE DEFINITIONS
=============================================================================*/


//! TUNERSTATE is used by the application and some API functions
typedef union tagTUNERSTATE 
{
    unsigned char   State;
    struct tagTUNERSTATE_bits
    {
        unsigned char AFC         :1; // OUT: TRUE if tuned, FALSE, if not
        unsigned char StereoSignal:1; // OUT: TRUE if stereo is received, FALSE if not
        unsigned char DX          :1; // IN:  TRUE  -> no attenuation, 
                                      //      FALSE -> attenuation
                                      //      This flag has no effect with the TerraTV Radio+
        unsigned char StereoMode  :1; // IN:  TRUE  -> receive stereo if possible, 
                                      //      FALSE -> force mono
        unsigned char Level       :2; // IN:  0 (very sensitive) .. 3 (insensitive), 
                                      //      only used for search commands
    }bits;
}TUNERSTATE;

typedef struct tagSTATION_DESCRIPTOR
{
    char   szStationName[16+1]; // RDS name has 8 chars max.
    ULONG  ulFrequencyHz;
    USHORT usPiCode;
}STATION_DESCRIPTOR,*PSTATION_DESCRIPTOR;



/*-----------------------------------------------------------------------------
   All export symbols are declared as extern "C"
   to prevent the C++ name decoration 
-----------------------------------------------------------------------------*/
#ifdef __cplusplus 
extern "C"
{
#endif


/*=============================================================================
    EXPORTED SYMBOLS
=============================================================================*/
                        
/*-----------------------------------------------------------------------------
   Event handles used by the application
   These events are used to notify the application about new RDS data
-----------------------------------------------------------------------------*/
    extern RDSLIBAPI HANDLE ghEventCT;       //!< new time signal
    extern RDSLIBAPI HANDLE ghEventRT;       //!< new radiotext line
    extern RDSLIBAPI HANDLE ghEventPS;       //!< program name complete
    extern RDSLIBAPI HANDLE ghEventNewGroup; //!< a new RDS block was received

/*=============================================================================
   EXPORTED FUNCTIONS
=============================================================================*/

    /*! Call radioInitialize() before any other API function */
    RDSLIBAPI TTRESULT radioInitialize(ULONG ulReserved); 
    RDSLIBAPI TTRESULT radioUninitialize();

   
     /*----------------------------------------------------------------------------
          Configuration
     -----------------------------------------------------------------------------*/

    //! Hardware IDs
    typedef enum tagTTRADIO_HWID{
         TTRADIO_INVALID_HARDWARE   =  0,  //!< ### unknown or uninitialized ###
         TTRADIO_DIRECTSHOW         =  1,  //!< ### generic DirectShow support ###
         TTRADIO_VOBIS_ISA          =  2,  //!< ISA Highscreen Radio-Boostar (as AR ISA but w/o RDS)
         TTRADIO_ACTIVERADIO_ISA    =  3,  //!< ISA TerraTec ActiveRadio Standalone 
         
         TTRADIO_AR_UPGRADE_BOOSTAR =  4,  //!< ISA Highscreen Sound-Boostar (similar to Base1)
         TTRADIO_AR_UPGRADE_BASE1   =  5,  //!< ISA TerraTec Base1/64
         TTRADIO_AR_UPGRADE_EWS64S  =  6,  //!< ISA TerraTec EWS64S 
         TTRADIO_AR_UPGRADE_XLERATE =  7,  //!< PCI TerraTec XLerate 
         TTRADIO_AR_UPGRADE_DMX     =  8,  //!< PCI TerraTec SoundSystem DMX 

         TTRADIO_TERRATV_RADIO_WDM  =  9,  //!< PCI TerraTV Radio+        (WDM drivers)
         TTRADIO_TERRATV_WDM        = 10,  //!< PCI TerraTV+ + AR Upgrade (WDM drivers)
         TTRADIO_TVALUE_RADIO_WDM   = 11,  //!< PCI TValue Radio mono, no RDS (WDM drivers)
         TTRADIO_CINERGY_600_MK2    = 12,  //!< PCI RDS, SAA7134  FM12MK2 (WDM/BDA drivers)
         TTRADIO_CINERGY_600_MK3    = 13,  //!< PCI RDS, SAA7134, FM12MK3 (WDM/BDA drivers)
         TTRADIO_CINERGY_250_PCI    = 14,  //!< PCI SAA7131, silicon tuner(BDA drivers)
         TTRADIO_NUM_HWIDS
    }TTRADIO_HWID; 
  
    
    typedef enum  tagTTRADIO_BUSTYPE
    {
         TTRADIO_BUS_UNKNOWN = 0,
         TTRADIO_BUS_ISA,
         TTRADIO_BUS_PCI,
         TTRADIO_BUS_USB,
         TTRADIO_BUS_IEEE1394,
         TTRADIO_NUM_BUSTYPES 
    }TTRADIO_BUSTYPE;
   


/*----------------------------------------------------------------------------
    Radio capabilities (RADIOCAPS::dwCaps)
-----------------------------------------------------------------------------*/
#define TTRADIO_CAP_IS_UPGRADE    0x00000001 /*!< The radio is an addon to the main device.
                                                  Radio function may not always be available.*/
#define TTRADIO_CAP_I2C_INTERFACE 0x00000002 //!< Device exports an I2C interface
#define TTRADIO_CAP_RDS           0x00000004 //!< Device has a RDS decoder

#define TTRADIO_CAP_HW_MUTE       0x00000100 //!< Volume: can mute hardware 
#define TTRADIO_CAP_HW_VOLUME     0x00000200 //!< Volume: can set volume on hardware
#define TTRADIO_CAP_SHARED_VOLUME 0x00000400 //!< Volume: TV cards only: TV and FM uses the same volume switches

#define TTRADIO_CAP_STEREO        0x00010000 //!< Tuner: is a stereo device
#define TTRADIO_CAP_FORCE_MONO    0x00020000 //!< Tuner: can switch mono/stereo
#define TTRADIO_CAP_INPUT_ATT     0x00040000 //!< Tuner: switchable input attenutation (local mode)(OM5610)
#define TTRADIO_CAP_SIGNALQUALITY 0x00080000 //!< Tuner: can return signals strength values
#define TTRADIO_CAP_DIRECTSHOW    0x00100000 //!< Tuner: TV cards only: uses the DirectShow interface for control
#define TTRADIO_CAP_SHARED_TUNER  0x00200000 //!< Tuner: TV cards only: combined TV/FM TUNER

   
/*=============================================================================
    Device 
=============================================================================*/
    typedef struct tagRADIOCAPS
    {
        int             nSerialId;    // serial ID# read from device (usually 0)
        TTRADIO_HWID    HardwareId;
        TTRADIO_BUSTYPE BusType;
        char            szDesc[128];  // friendly name
        ULONG           dwCaps;       // see TTRADIO_xxx flags
    }RADIOCAPS;

    RDSLIBAPI int      radioGetNumDevs ();
    RDSLIBAPI TTRESULT radioGetDevCaps (IN int nIdx,OUT RADIOCAPS* pCaps);

    
/*=============================================================================
    FM tuner
=============================================================================*/
    /*! Forces a stereo program to mono mode. 
    This can reduce the noise level at bad reception.*/
    RDSLIBAPI TTRESULT radioForceMono(IN UCHAR ucDevIdx,BOOL fMono);

    /*! Attenuates the input (HF) signal. Too strong HF signal level
    can lead to distorted audio. This function is also often called "Local".
    (unattenauated - "DX") */
    RDSLIBAPI TTRESULT radioSetInputAttenuation(IN UCHAR ucDevIdx,BOOL fLocal);


    /*! radioGetState returns info if the station is stereo and if the AFC is locked
        For cards with combined FM/TV tuner, Stereo and Lock is always TRUE*/
   RDSLIBAPI TTRESULT radioGetState    (IN UCHAR ucDevIdx,IN TUNERSTATE* psState);

   /*! Take care: radioGetFrequency() doesn't always return correct values.
       radioGetFrequency()is not supported by the TerraTV Radio+ !!! */
   RDSLIBAPI TTRESULT radioGetFrequency(IN UCHAR ucDevIdx,OUT PULONG pulFreqHz);
   RDSLIBAPI TTRESULT radioSetFrequency(IN UCHAR ucDevIdx,IN   ULONG  ulFreqHz);


   enum { SEARCH_DOWN,SEARCH_UP};
   /*! SEARCH
   NOTE: radioSearch stops RDS before the search and start it again afterwards */
   RDSLIBAPI TTRESULT radioSearch      (IN UCHAR ucDevIdx,
       IN OUT PULONG pulFreqHz, // IN: must be filled with start frequency
       BOOL fSearchUp,          // SEARCH_UP or SEARCH_DOWN
       TUNERSTATE* pTs);

/*---------------------------------------------------------------
    Volume control functions
---------------------------------------------------------------*/
    RDSLIBAPI TTRESULT VolumeMute(IN UCHAR ucDevIdx,BOOL fMute);
    
    // Parameter: wNewVolume [0..127]
#define TTRADIO_MAX_VOLUME 127
    RDSLIBAPI TTRESULT VolumeSet(IN UCHAR ucDevIdx,UCHAR ucVolume);


/*---------------------------------------------------------------
    Miscellaneous
---------------------------------------------------------------*/
	// \return: the hardware ID of the currently selected device
    RDSLIBAPI TTRESULT radioGetDeviceType(IN UCHAR ucDevIdx,TTRADIO_HWID* pHwId);   

	//! Queries an internal PCI capture card table
    RDSLIBAPI TTRADIO_HWID radioGetDeviceTypeFromSsid(USHORT usSubvendorId,USHORT usSubdeviceId);   


    //! Generic IIC access (if available for this device)
    RDSLIBAPI TTRESULT radioWriteI2CBuffer(IN UCHAR ucDevIdx,IN UCHAR ucAddress, OUT PUCHAR pucBuffer,IN int nSize);    
    RDSLIBAPI TTRESULT radioReadI2CBuffer (IN UCHAR ucDevIdx,IN UCHAR ucAddress, IN  PUCHAR pucBuffer,IN int nSize);    
    
    //! \return: the full path to the hardware detection log file
    RDSLIBAPI LPCSTR   radioGetLogfile(void);    


/*-----------------------------------------------------------------------------
    Station list
-------------------------------------------------------------------------------
    FMB NOTE 13-Dec-04: 
       For now the station list of the ActiveRadio
       software is utilized. Stations are only available
       if a scan was run form the ActiveRadio software.
-----------------------------------------------------------------------------*/
    //! \return: <0 if failed
    RDSLIBAPI int      radioGetNumStations(void);    
    RDSLIBAPI TTRESULT radioGetStationListItem (IN int nIdx,OUT PSTATION_DESCRIPTOR pDesc);    

/*-----------------------------------------------------------------------------
     End of extern "C" part
-----------------------------------------------------------------------------*/
#ifdef __cplusplus
}       // extern "C"
#endif

/*-----------------------------------------------------------------------------
     RDS API
-----------------------------------------------------------------------------*/
#include "TTApi_RDS.h" // include RDS functions

#endif //__IOEXPORT_H__

