/*=============================================================================
   FILE:         Ar_Ioctl.h
-------------------------------------------------------------------------------
   DESCRIPTION:  ActiveRadio NT driver interface (TTRadio.sys)
   AUTHOR:       F.M.Birth
   HISTORY:      03-Aug-1999
                 14-Dec-2000 - removed #include directives
				             - added some struct definitions
   NOTES:        I/O control codes for use with DeviceIoControl
                 If there are more than one radio cards in the system,
                 IOCTL_FIND_RADIO should be called to get info about 
				 number and type of these cards

-------------------------------------------------------------------------------
   COPYRIGHT:    (c) 1998-2004, TerraTec Electronic GmbH. All Rights Reserved.

                 Information in this file is the intellectual property of
                 TerraTec Electronic GmbH, and contains trade secrets
                 that must be stored and viewed confidentially.
                 By viewing, using, modifying and compiling 
                 the programming code below you acknowledge you have read,
                 accepted and executed the Software licensing agreement 
                 your company has established with TerraTec Electronic GmbH.
=============================================================================*/
#ifndef __AR_IOCTL_H__
#define __AR_IOCTL_H__

#include "TTApi_FM.h"

#pragma pack( push, enter_ar_ioctl,8) // required for ActiceRadio driver interface (AR_Icotl.h)


#define AR_DRIVER_NAME   L"TTRadio"
#define AR_DEVICE_NAME_0 "\\\\.\\ActiveRadio0"

#define AR_TYPE 40000           //! Device type -> in the "User Defined" range."

#define RDS_BUFSIZE			 20	//! size of RDS data buffer
#define RDS_STATE_VALUES	200	//! build the average state over x blocks
							    // => after x blocks (= x/4 groups) a Buffer Full Event is set

#define AR_WINNT_MAX_NUM_CARDS    8 //! max. number of ActiveRadio ISA cards in one system




/*=============================================================================
	RDS structures
=============================================================================*/



/*---------------------------------------------------------------
	SAA6588_READ - 7 bytes input
---------------------------------------------------------------*/
typedef union RDSTAT
{   			
 	unsigned char byte;			
	struct RDSTATBF
 	{
		unsigned char ELB 	:2; // error status of last processed block
		unsigned char RSTDC	:1;	// (no) reset detected
		unsigned char DOFL	:1;	// (no) data overflow
		unsigned char SYNC	:1;  // (not) synchronized
		unsigned char BL		:3;	// block identification number	
 	 }bits;
}RDSTAT; 

typedef union tagRDESTA
{   			
 	unsigned char byte;			
	struct RDESTABF
 	{
		unsigned char EPB	:2; // error status of previously processsed block
		unsigned char BEC	:6;	// number of counted block errors
 	 }bits;
}RDESTA; 

typedef union tagRDSGQL
{   			
 	unsigned char byte;			
	struct RDSGQLBF
 	{
		unsigned char SQI	:5; // signal quality indication (0..15)
		unsigned char BP		:3;	// block identification number of 
						//   previously processed block
 	 }bits;
}RDSGQL; 


/*---------------------------------------------------------------
	HW - SAA6588 routines
---------------------------------------------------------------*/
typedef struct tagIIC_READ	
{   						
	RDSTAT		   RDSTAT;
	unsigned short LastBlock;
	unsigned short PrevBlock;
	RDESTA		   RDESTA;
	RDSGQL		   RDSGQL;
}SAA6588_READ; 







/*=============================================================================
	Common structures
=============================================================================*/



/*---------------------------------------------------------------
	Hardware caps and state
---------------------------------------------------------------*/	
typedef struct tagHW_INSTANCE_INFO
{
    // hardware description
	TTRADIO_HWID    HW_ID;
	unsigned char   VolMode;     // Volume control mode
    TTRADIO_BUSTYPE Bus;         // Bus Type

	unsigned long   dwRadioPort; // Tuner control port
	unsigned long   dwVolPort;   // Hardware volume control port

	char            szVendorId[9]; // 
	char            szDeviceId[5]; // 
	char            CSN;		    // Card Select No (only used for ISAPNP)

    // Current setttings
	unsigned long   dwCurFreqHz;   // Current frequency in Hz
	unsigned char   gfMute;
	unsigned char   gfStereo;
	unsigned short  gwVolume;

}HW_INSTANCE_INFO,*PHW_INSTANCE_INFO;

// FMB NOTE: structure in kernel mode must not be bigger than 4KB
// because otherwise the compiler inserts stack checking (__chkstk)
// which the linker cannot resolve in kernel mode.
typedef struct tagHW_DATA
{
    int               nReserved; 
	int               nNoCards;
	HANDLE            hDriver  [AR_WINNT_MAX_NUM_CARDS]; // Driver handles
	HW_INSTANCE_INFO  HwInst   [AR_WINNT_MAX_NUM_CARDS];
}HW_DATA;


/*---------------------------------------------------------------
	OM5610 - SHIFT REGISTER - 25 bit wide
---------------------------------------------------------------*/	
typedef union tagSR_TYPE	// accesses the shift register either as
{   						// ULONG or bit field
 	unsigned long		SR;			
	struct SRBF
 	{
		int F0_14   :15;// frequency
		int X15		:1;	// dummy
		int S16_17	:2;	// search-level
		int X18		:1;	// not used
		int P19		:1;	// local/DX
		int B20_21	:2;	// FM band selection
 	 	int M22		:1; // mono/stereo
 	 	int D23		:1;	// search up/down
 	 	int S24		:1;	// search start/end
 	 	int X25_31	:7; // not used
 	}bits;
}SR_TYPE; 




/*=============================================================================
	I/O control codes
=============================================================================*/
// The IOCTL function codes from 0x800 to 0xFFF are for customer use.

/*-----------------------------------------------------------------------------
	IOCTL_FIND_RADIO
-------------------------------------------------------------------------------
	DESCRIPTION: Auto-detection of a card with a radio tuner
	INPUT:       ---
	OUTPUT:      HW_DATA structure containing configuration info about all installed radios
-----------------------------------------------------------------------------*/
#define IOCTL_FIND_RADIO           CTL_CODE(AR_TYPE,  0x900, METHOD_BUFFERED, FILE_READ_ACCESS)

/*-----------------------------------------------------------------------------
	OM6510 read/write status register
-----------------------------------------------------------------------------*/
#define IOCTL_OM5610_SET_FREQUENCY CTL_CODE(AR_TYPE,  0x910, METHOD_BUFFERED, FILE_WRITE_ACCESS)
#define IOCTL_OM5610_GET_FREQUENCY CTL_CODE(AR_TYPE,  0x911, METHOD_BUFFERED, FILE_READ_ACCESS)
#define IOCTL_OM5610_GET_STATE     CTL_CODE(AR_TYPE,  0x912, METHOD_BUFFERED, FILE_READ_ACCESS)
#define IOCTL_OM5610_SEARCH        CTL_CODE(AR_TYPE,  0x913, METHOD_BUFFERED, FILE_READ_ACCESS)

/*-----------------------------------------------------------------------------
	SAA6588 IIC read/write 
-----------------------------------------------------------------------------*/
// IN: 3 bytes
#define IOCTL_SAA6588_WRITE_IIC    CTL_CODE(AR_TYPE,  0x920, METHOD_BUFFERED, FILE_WRITE_ACCESS)
// OUT: 7 bytes
#define IOCTL_SAA6588_READ_IIC     CTL_CODE(AR_TYPE,  0x921, METHOD_BUFFERED, FILE_READ_ACCESS)

/*-----------------------------------------------------------------------------
	Set volume via hardware (Standalone card,DMX w/o ESS driver)
-----------------------------------------------------------------------------*/
/* IN:  Notes   new volume value [0..127] (2 bytes)
*/
#define IOCTL_HW_SET_VOLUME        CTL_CODE(AR_TYPE,  0x930, METHOD_BUFFERED, FILE_WRITE_ACCESS)

/*-----------------------------------------------------------------------------
	DMX: is the sound driver loaded
-----------------------------------------------------------------------------*/
#define IOCTL_DMX_IS_DRIVER_LOADED CTL_CODE(AR_TYPE,  0x931, METHOD_BUFFERED, FILE_WRITE_ACCESS)

/*-----------------------------------------------------------------------------
	Generic IIC commands
-----------------------------------------------------------------------------*/
#define IOCTL_IIC_WRITE           CTL_CODE(AR_TYPE,  0x952, METHOD_BUFFERED, FILE_WRITE_ACCESS)
#define IOCTL_IIC_READ            CTL_CODE(AR_TYPE,  0x953, METHOD_BUFFERED, FILE_READ_ACCESS)


#pragma pack( pop, enter_ar_ioctl)

#endif __AR_IOCTL_H__
