/*! \file 
===============================================================================
   FILE:         TTApi_RDS.h
-------------------------------------------------------------------------------
   DESCRIPTION:  ActiveRadio RDS API
                 Hardware control functions & structures
   AUTHOR:       F.M.Birth, TerraTec Electronic GmbH
-------------------------------------------------------------------------------
   HISTORY:      19-Oct-04 FB: Separated RDS API from FM tuner API
   NOTES:        
-------------------------------------------------------------------------------
   COPYRIGHT:    (c) 2004, TerraTec Electronic GmbH.
                 (c) 2004, Friedhelm Birth
=============================================================================*/

#ifndef __RDS_API_H__
#define __RDS_API_H__


#define RDS_RT_LENGTH 64 // 64 characters (excluding the null-termination character)
#define RDS_PS_LENGTH  8 //  8 characters (excluding the null-termination character)

/*===============================================================
    RDS STRUCTURES
===============================================================*/   

/*---------------------------------------------------------------
    Country code (CC)
---------------------------------------------------------------*/
typedef union tagRDS_CC {       // country code (group type 1)
    USHORT word;
    struct tagCC_bits {
        USHORT ECC_LSB :4;        
        USHORT ECC_MSB :4;        
        USHORT PI_MSB  :4;        
    }bits;
}RDS_CC;

/*---------------------------------------------------------------
    Pin code (PIN)
---------------------------------------------------------------*/

typedef union tagRDS_PIN {
    USHORT word;
    struct tagPIN_bits {
        USHORT minute :6;     // 6 bit
        USHORT hour   :5;     // 5 bit
        USHORT day    :5;     // 5 bit
    }bits;
}RDS_PIN;

/*---------------------------------------------------------------
    Time signal (CT)
---------------------------------------------------------------*/
typedef struct tagRDS_TIME
{
    UCHAR        KW;             // 6 bit calendar week;
    UCHAR        year;           // 7 bit; years since 1900; e.g. year==104 for 2004
    UCHAR        month;          // 4 bit
    UCHAR        day;            // 5 bit
    UCHAR        weekday;        // 3 bit
    UCHAR        hour;           // 5 bit
    UCHAR        minute;         // 6 bit
}RDS_TIME;

/*---------------------------------------------------------------
    Linkage Set Number (LSN)
---------------------------------------------------------------*/
typedef union tagRDS_LSN   // Linkage Set Number used for EON
 {
    USHORT word;
    struct
    {
        USHORT LSN    :4; // Linkage Set Number
        USHORT ILS    :1; // International Linkage Set Identifier
        USHORT NU     :1; // not used
        USHORT EG     :1; // artverwandte Sendebereichscodes
        USHORT LA     :1; // Link Actuator
    }bits;
}RDS_LSN;

/*---------------------------------------------------------------
    EON
---------------------------------------------------------------*/
//! NOTE: ON - other network, TN - this network
typedef struct tagRDS_EON
{
    char     PS_ON[RDS_PS_LENGTH+1];
    UCHAR    PTY_ON;
    UCHAR    TA_ON;
    UCHAR    TP_ON;
    RDS_PIN PIN_ON;
    RDS_LSN LSN;
    
    // Verfahren A
    ULONG   FREQ_AF_ON[2]; 

    // Frequenzkartei-Verfahren:
    ULONG   FREQ_TN[5];    // Abstimmfrequenz (this network)
    ULONG   FREQ_ON[5];    // AF{4] is a LW/MW frequency

}RDS_EON;

/*---------------------------------------------------------------
    RDS_DATA
---------------------------------------------------------------*/
#define RDS_MAX_NUM_AF 25

typedef struct tagRDS_DATA
{           
    UCHAR          PTY;         // Program type              (5 bit)
    UCHAR          TP;          // Traffic program           (1 bit)
    UCHAR          TA;          // Traffic announcement      (1 bit)
    UCHAR          DI;          // Decoder control           (4 bit)
    UCHAR          DI_chk;      // Decoder control checkword (4 bit)
    UCHAR          M_S;         // Musik-/Sprache-Kennnung   (1 bit)
    UCHAR          RT_Flag;     // Flag changes, if a new text line is sent (1 bit)
                                //   (unfortunately not used by all stations)
    UCHAR          NumAf;       // Size of Alternative Frequencies (AF) list (list entries)
    ULONG          Af[RDS_MAX_NUM_AF];   //!< AF array (in Hz)
    RDS_CC         CC;          // Country code 
    RDS_PIN        PIN;         // Programmbeitragskennung
    RDS_TIME       CT;          // Time signal
    RDS_EON        EON;         // EON
    USHORT         PI;          // PI code
    char           PS    [RDS_PS_LENGTH+1];  // Station name              (8 chars)
    char           RT    [RDS_RT_LENGTH+1];  // Radiotext                 (64 chars)
    char           RT_bak[RDS_RT_LENGTH+1];  // backup of last complete RT string Radiotext
    char           RT_tmp[RDS_RT_LENGTH+1];  // temporary RT buffer to check for identical RT lines
}RDS_DATA; 



/*=============================================================================
    RDS API functions
=============================================================================*/

/*-----------------------------------------------------------------------------
   All export symbols are declared as extern "C"
   to prevent the C++ name decoration 
-----------------------------------------------------------------------------*/
#ifdef __cplusplus 
extern "C"
{
#endif


    /*---------------------------------------------------------------
        RDS Decoder control functions
    ---------------------------------------------------------------*/

    //! Starts the timer and RDS data collection; old RDS data are deleted
    RDSLIBAPI TTRESULT RDS_Start(IN UCHAR ucDevIdx);
    //! Stops the timer
    RDSLIBAPI TTRESULT RDS_Stop (IN UCHAR ucDevIdx);
    
    //! Restarts synchronization
    RDSLIBAPI TTRESULT RDS_Resync(IN UCHAR ucDevIdx); 

    //! This function checks if a PI code can be received. Other RDS data are kept intact.
    //! \param: OUT PWORD pwPiCode - valid PI code, or 0x0000 if station has no RDS or RDS cannot be received
    RDSLIBAPI TTRESULT RDS_GetPiCode(IN UCHAR ucDevIdx, OUT USHORT* pusPiCode);


    //! This function transfers the current RDS buffer to the application
    RDSLIBAPI TTRESULT RDS_EvalData (IN UCHAR ucDevIdx,OUT RDS_DATA* pRdsData);

    /*---------------------------------------------------------------
        RDS Decoder - Raw data
    ---------------------------------------------------------------*/
typedef struct tagRDS_GROUP
{
	unsigned char		group;	 // group type
	unsigned char		version; // group version A or B
	unsigned short		block[4];// block data
	unsigned char		error[4];// error status
}RDS_GROUP;
    
    /*! Get Raw Data: One RDS group consists of 4 Blocks. Blocks are transmitted every 20 msec
        -> ca 13 RDS groups/sec */
    RDSLIBAPI TTRESULT RDS_GetRawData(IN UCHAR ucDevIdx,
        OUT RDS_GROUP* pRdsData,              //!< \param: buffer to receive RDS groups
        IN  int        nSize,                 //!< \param: buffer size of pRdsData
        OUT PUSHORT    pusNumGroupsReturned); //!< \param: number of groups filled by this routine


    /*===============================================================
        SAA6588 special handling
    ===============================================================*/
    typedef union tagSAA6588_STATE
    {
        USHORT State;
        struct tagSAA_bits {
            USHORT    SQI     :4; // signal quality 0 ..15    (4 bit) 
            USHORT    BEC     :7; // block error count 0..128 (6 bits) 
            USHORT    DOFL    :1; // data overflow            (1 bit)
            USHORT    DAC     :2; // data output mode 0.. 2   (2 bit) - ignored !!!
            USHORT    SYNC    :1; // sync                     (1 bit)
        } bits;
    }SAA6588_STATE;                                 // =>13 bit
    /*---------------------------------------------------------------
        SAA6588 control functions
    ---------------------------------------------------------------*/
    //! Retrieves data about the current SAA6588 reception and error count state
    RDSLIBAPI TTRESULT RDS_SAA6588_GetDecoderState(IN UCHAR ucDevIdx,SAA6588_STATE* pSaaState);


    /*===============================================================
        RDS logging
        For internal use only. Do not use !!!
    ===============================================================*/
#define RDSLOG_RT  0x00000001 // log radio text
#define RDSLOG_AF  0x00000002 // log alternative frequencies
#define RDSLOG_EON 0x00000004 // log EON data
#define RDSLOG_SAA 0x00010000 // log decoder state

    //! These functions were used for debugging and are not yet completely tested
    //!\param: ulFlags: Combination of RDSLOG_xxx flags
    RDSLIBAPI TTRESULT RDS_StartLog(IN UCHAR ucDevIdx,IN LPCSTR lpszPath,IN BOOL fAppend, IN ULONG ulFlags);
    RDSLIBAPI TTRESULT RDS_StopLog (IN UCHAR ucDevIdx);




#ifdef __cplusplus
}       // extern "C"
#endif

#endif // __RDS_API_H__
