/*=============================================================================
   FILE:         MainWnd.cpp
-------------------------------------------------------------------------------
   DESCRIPTION:  CMainWnd class implementation
   AUTHOR:       F.M.Birth
   HISTORY:      5-8-1999
   NOTES:        
-------------------------------------------------------------------------------
   COPYRIGHT:    (c) 1998-2005, TerraTec Electronic GmbH. All Rights Reserved.

                 Information in this file is the intellectual property of
                 TerraTec Electronic GmbH, and contains trade secrets
                 that must be stored and viewed confidentially.
                 By viewing, using, modifying and compiling 
                 the programming code below you acknowledge you have read,
                 accepted and executed the Software licensing agreement 
                 your company has established with TerraTec Electronic GmbH.
=============================================================================*/
#include <Windows.h>
#include <CrtDbg.h>

#include "Resource.h"
#include "TTApi_FM.h"
#include "TTApi_RDS.h"
#include "MainWnd.h"
#include "Main.h"

#define FREQ_LINE ( 50*ONE_KHZ)
#define FREQ_PAGE (500*ONE_KHZ)
 
//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CMainWnd::CMainWnd()
{

	m_hTimer  = 0;
    m_nDevIdx = 0;
}

CMainWnd::~CMainWnd()
{
	radioUninitialize();
}
 

/*=============================================================================
  Function    : CMainWnd::OnInitDialog
-------------------------------------------------------------------------------
  Description : 
  Returns     : void  -> 
-------------------------------------------------------------------------------
  Notes       : 
=============================================================================*/
void CMainWnd::OnInitDialog()
{ 

	m_hWnd = FindWindow(NULL,"ActiveRadio API Sample");

	TunerState.bits.StereoMode	= STEREO;
	TunerState.bits.DX			= MODE_DX;
	TunerState.bits.Level		= 1;
	m_dwFreq					= FREQ_DEF;
	m_nVolume					= VOL_DEF;


	CheckDlgButton(m_hWnd,IDC_LEVEL_0+TunerState.bits.Level,BST_CHECKED);
	
	SCROLLINFO SI;
	SI.cbSize = sizeof (SCROLLINFO);
	SI.fMask  = SIF_PAGE|SIF_POS|SIF_RANGE;
	SI.nMin   = FREQ_MIN;
	SI.nMax   = FREQ_MAX;
	SI.nPage  = 5000;
	SI.nPos   = m_dwFreq;
	SetScrollInfo(GetDlgItem(Dlg.m_hWnd,IDC_SLIDER_FREQ),SB_CTL,&SI,TRUE);

	SI.nMin   = VOL_MIN;
	SI.nMax   = VOL_MAX;
	SI.nPage  = 10;
	SI.nPos   = m_nVolume;
	SetScrollInfo(GetDlgItem(Dlg.m_hWnd,IDC_SLIDER_VOL),SB_CTL,&SI,TRUE);

	/*----------------------------------------------------------------------------
	// Check for multiple cards and init the combobox
	-----------------------------------------------------------------------------*/
	TTRESULT ttr = radioInitialize(NULL);
    _ASSERT(TTSUCCEEDED(ttr));

    int nNoCards = radioGetNumDevs();
    
    HWND hWnd = GetDlgItem(m_hWnd,IDC_COMBO_SELECT_CARD);
    for (int i=0;i<nNoCards;i++)
    {
        //sprintf(szString,"Card %d",i);
        RADIOCAPS rc;
        radioGetDevCaps(i,&rc);
        LRESULT lRes = SendMessage(hWnd,CB_ADDSTRING,0,(LPARAM)rc.szDesc);
        _ASSERT(lRes != LB_ERR);
    }
    
    SendMessage(hWnd,CB_SETCURSEL,0,0);

	// Init the hardware	
	if (nNoCards>0)
		OnNewHardware(0);
}


/*=============================================================================
  Function    : OnNewHardware
-------------------------------------------------------------------------------
  Description : 
  Returns     : BOOL  -> 
  Parameters  : nListIdx -> new Radio card index
-------------------------------------------------------------------------------
  Notes       : 
=============================================================================*/
BOOL CMainWnd::OnNewHardware(int nListIdx)
{
	if (m_hTimer)
	{
		KillTimer(m_hWnd,m_hTimer);
		m_hTimer = NULL;
	}
	
    m_nDevIdx = nListIdx;

	VolumeMute  (m_nDevIdx,FALSE);
	VolumeSet   (m_nDevIdx,m_nVolume);
	SetFrequency(m_dwFreq,&TunerState);
	
	m_hTimer = (UINT)m_hWnd;
	SetTimer(m_hWnd,m_hTimer,RDS_UPDATE_PERIOD,NULL);

	return TRUE;
}
	



/*=============================================================================
  Function    : CMainWnd::OnSliderFreq
-------------------------------------------------------------------------------
  Description : 
  Returns     : void  -> 
  Parameters  : int nScrollCode -> 
              : int nPos -> 
-------------------------------------------------------------------------------
  Notes       : 
=============================================================================*/
void CMainWnd::OnSliderFreq(int nScrollCode, int nPos)
{
	SCROLLINFO SI;
	SI.cbSize = sizeof(SCROLLINFO);
	switch(nScrollCode)
	{
	case SB_LEFT:     m_dwFreq = FREQ_MIN; break;
	case SB_RIGHT:    m_dwFreq = FREQ_MAX; break;

	case SB_LINELEFT:
		if (m_dwFreq > (FREQ_MIN+FREQ_LINE))
			m_dwFreq -= FREQ_LINE;
		break;
	case SB_LINERIGHT:
		if (m_dwFreq < FREQ_MAX-FREQ_LINE)
			m_dwFreq+=FREQ_LINE;
		break;
	case SB_PAGELEFT:
		if (m_dwFreq > (FREQ_MIN+FREQ_PAGE))
			m_dwFreq -= FREQ_PAGE;
		break;
	case SB_PAGERIGHT:
		if (m_dwFreq < FREQ_MAX-FREQ_PAGE)
			m_dwFreq+=FREQ_PAGE;
		break;
	case SB_THUMBPOSITION:
		break;
	case SB_THUMBTRACK:
		SI.fMask = SIF_TRACKPOS;
		GetScrollInfo(GetDlgItem(Dlg.m_hWnd,IDC_SLIDER_FREQ),SB_CTL,&SI);
		m_dwFreq = SI.nTrackPos;
		DB("Freq = %ld",m_dwFreq);

		break;
	case SB_ENDSCROLL:
		SI.nPos = m_dwFreq;
		DB("Freq Endscroll = %ld",m_dwFreq);
		SI.fMask = SIF_POS;
		SetScrollInfo(GetDlgItem(Dlg.m_hWnd,IDC_SLIDER_FREQ),SB_CTL,&SI,TRUE);

		break;
	}

	SetFrequency(m_dwFreq, &TunerState);
	
}



/*=============================================================================
  Function    : CMainWnd::OnSliderVol
-------------------------------------------------------------------------------
  Description : 
  Returns     : void  -> 
  Parameters  : int nScrollCode -> 
              : int nPos -> 
-------------------------------------------------------------------------------
  Notes       : 
=============================================================================*/
#define VOL_LINE  8
#define VOL_PAGE 16

void CMainWnd::OnSliderVol(int nScrollCode, int nPos)
{
	SCROLLINFO SI;
	SI.cbSize = sizeof(SCROLLINFO);
	switch(nScrollCode)
	{
	case SB_LINELEFT:
		if (m_nVolume > (VOL_MIN+VOL_LINE))
			m_nVolume -= VOL_LINE;
		break;
	case SB_LINERIGHT:
		if (m_nVolume < VOL_MAX-VOL_LINE)
			m_nVolume+=VOL_LINE;
		break;
	case SB_PAGELEFT:
		if (m_nVolume > (VOL_MIN+VOL_PAGE))
			m_nVolume -= VOL_PAGE;
		break;
	case SB_PAGERIGHT:
		if (m_nVolume < VOL_MAX-VOL_PAGE)
			m_nVolume+=VOL_PAGE;
		break;
	case SB_THUMBTRACK:
		SI.fMask = SIF_TRACKPOS;
		GetScrollInfo(GetDlgItem(Dlg.m_hWnd,IDC_SLIDER_VOL),SB_CTL,&SI);
		m_nVolume = SI.nTrackPos;
		break;
	case SB_ENDSCROLL:
		SI.nPos = m_nVolume;
		SI.fMask = SIF_POS;
		SetScrollInfo(GetDlgItem(Dlg.m_hWnd,IDC_SLIDER_VOL),SB_CTL,&SI,TRUE);
		break;
	}

	VolumeSet(m_nDevIdx,m_nVolume);
}



/*=============================================================================
  Function    : CMainWnd::SetFrequency
-------------------------------------------------------------------------------
  Description : 
  Returns     : void  -> 
  Parameters  : DWORD dwFreq -> 
              : TUNERSTATE* pTunerState -> 
-------------------------------------------------------------------------------
  Notes       : 
=============================================================================*/
void CMainWnd::SetFrequency(DWORD dwFreqHz, TUNERSTATE* pTunerState)
{
	TCHAR		szBuffer[256];
	DWORD		dwFreqRead;
	SCROLLINFO	SI;
	TTRESULT	Ret;
	
	// stop RDS polling before a tuner command since the different 
	// protocols might interfere with each other
	RDS_Stop(m_nDevIdx);
	Ret = radioSetFrequency(m_nDevIdx,dwFreqHz);
	
	Sleep(20); // Give the AFC some time to settle
	
	// FMB NOTE: Better check the returned frequency: it may be crap
    Ret = radioGetFrequency(m_nDevIdx,&dwFreqRead);//
	
	RDS_Start(m_nDevIdx);

	sprintf(szBuffer,"%3.2f MHz",((float)m_dwFreq)/ONE_MHZ);
	SetWindowText(GetDlgItem(m_hWnd,IDC_EDIT_FREQ_SET),szBuffer);
	if ((dwFreqRead < FREQ_MIN) || (dwFreqRead > FREQ_MAX) )
	     sprintf(szBuffer,"???");
    else sprintf(szBuffer,"%3.2f MHz",((float)dwFreqRead)/ONE_MHZ);
	SetWindowText(GetDlgItem(m_hWnd,IDC_EDIT_FREQ_READ),szBuffer);

	SI.cbSize = sizeof(SCROLLINFO);
	SI.nPos = m_dwFreq;
	SI.fMask = SIF_POS;
	SetScrollInfo(GetDlgItem(Dlg.m_hWnd,IDC_SLIDER_FREQ),SB_CTL,&SI,TRUE);
}


/*=============================================================================
  Function    : CMainWnd::OnSearch
-------------------------------------------------------------------------------
  Description : 
  Returns     : void  -> 
  Parameters  : BOOL fDir -> 
-------------------------------------------------------------------------------
  Notes       : 
=============================================================================*/
void CMainWnd::OnSearch(BOOL fDir)
{
	TTRESULT Ret;
	char szBuffer[150];
	DWORD dwFreqTmp = m_dwFreq;

AGAIN:
	Ret  = radioSearch(m_nDevIdx,&dwFreqTmp,fDir,&TunerState);
	switch (Ret)
	{
    case TTRADIO_E_NO_STATION_FOUND:
		strcpy(szBuffer,"No station found!");
		break;
	case TTRADIO_E_TOO_CLOSE:
		OutputDebugString("Search Error - too close: repeat search\n");
		sprintf(szBuffer,"Found the same station again!\nChange frequency and search again!");
		if (fDir = SEARCH_UP)
			 dwFreqTmp += 20*ONE_KHZ;
		else dwFreqTmp -= 20*ONE_KHZ;
		goto AGAIN;
		break;
    case TTRADIO_E_INVALID_FREQUENCY:
		sprintf (szBuffer, "Invalid frequency returned!");
		break;

	case TTRADIO_S_OK:
		m_dwFreq = dwFreqTmp;
        sprintf(szBuffer,"New station found at %3.2f MHz",((float)m_dwFreq)/ONE_MHZ);
		break;

	default:
        strcpy(szBuffer,"Error while searching");
	}

	MessageBox(m_hWnd,szBuffer,"Search",MB_OK);
	SetFrequency(m_dwFreq,&TunerState);

}




/*=============================================================================
  Function    : CMainWnd::OnTimer
-------------------------------------------------------------------------------
  Description : 
  Returns     : void  -> 
  Parameters  : int nIDEvent -> 
-------------------------------------------------------------------------------
  Notes       : 
=============================================================================*/
void CMainWnd::OnTimer(int nIDEvent)
{

	RDS_EvalData(m_nDevIdx,&m_RDS_Data);
	if (m_RDS_Data.PI == 0)
	{
		 SetWindowText(GetDlgItem(m_hWnd,IDC_EDIT_PS),"* No RDS *");
		 SetWindowText(GetDlgItem(m_hWnd,IDC_EDIT_RT),"* No RDS *");
	}
	else 
	{
		SetWindowText(GetDlgItem(m_hWnd,IDC_EDIT_PS),m_RDS_Data.PS);
		SetWindowText(GetDlgItem(m_hWnd,IDC_EDIT_RT),m_RDS_Data.RT);
	}
}
